/*
 * Decompiled with CFR 0.152.
 */
package com.jeremyzay.zaychess.controller.game;

import com.jeremyzay.zaychess.model.game.GameState;
import com.jeremyzay.zaychess.model.move.Move;
import com.jeremyzay.zaychess.model.move.MoveGenerator;
import com.jeremyzay.zaychess.model.move.MoveType;
import com.jeremyzay.zaychess.model.move.PromotionPiece;
import com.jeremyzay.zaychess.model.pieces.Piece;
import com.jeremyzay.zaychess.model.rules.GameOverType;
import com.jeremyzay.zaychess.model.util.PlayerColor;
import com.jeremyzay.zaychess.model.util.Position;
import com.jeremyzay.zaychess.services.application.history.MoveHistoryInMemory;
import com.jeremyzay.zaychess.services.application.history.MoveHistoryService;
import com.jeremyzay.zaychess.services.application.notation.NotationSAN;
import com.jeremyzay.zaychess.services.infrastructure.engine.EngineService;
import com.jeremyzay.zaychess.services.infrastructure.engine.SerendipityEngineService;
import com.jeremyzay.zaychess.services.infrastructure.network.MoveCodec;
import com.jeremyzay.zaychess.services.infrastructure.network.MoveMessage;
import com.jeremyzay.zaychess.services.infrastructure.network.NetworkTransport;
import com.jeremyzay.zaychess.view.gui.BoardPanel;
import com.jeremyzay.zaychess.view.gui.ChessFrame;
import com.jeremyzay.zaychess.view.gui.PromotionDialog;
import java.awt.Color;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

public class GameController
implements NetworkTransport.Listener {
    private final GameState gameState;
    private BoardPanel boardPanel;
    public List<String> moveLog = new ArrayList<String>();
    private final List<String> wireLog = new ArrayList<String>();
    private final MoveHistoryService history;
    private NetworkTransport transport;
    private volatile boolean networkReady = true;
    private PlayerColor localSide = null;
    private Position highlightedSquare = null;
    private Position selectedPosition = null;
    private EngineService engine = null;
    private volatile boolean engineThinking = false;
    private static final boolean RANK0_IS_BOTTOM = false;

    public GameController(GameState gameState) throws Exception {
        this(gameState, new MoveHistoryInMemory());
    }

    public GameController(GameState gameState, MoveHistoryService history) throws Exception {
        this.gameState = gameState;
        this.history = Objects.requireNonNullElseGet(history, MoveHistoryInMemory::new);
    }

    public void setLocalSide(PlayerColor side) {
        this.localSide = side;
    }

    public List<String> getWireLog() {
        return this.wireLog;
    }

    public void setBoardPanel(BoardPanel boardPanel) {
        this.boardPanel = boardPanel;
    }

    public GameState getGameState() {
        return this.gameState;
    }

    public BoardPanel getBoardPanel() {
        return this.boardPanel;
    }

    public NetworkTransport getHost() {
        return this.transport;
    }

    public boolean isUsingEngine() {
        return this.engine != null;
    }

    public void setEngine() {
        Path engineJar = GameController.appHome().resolve("engines").resolve("Serendipity.jar");
        this.engine = new SerendipityEngineService(GameController.javaCmd(), engineJar.toString());
        try {
            this.engine.start();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void attachNetwork(NetworkTransport t) {
        this.transport = t;
        this.networkReady = false;
        t.setListener(this);
        t.start();
    }

    public boolean isOnline() {
        return this.transport != null;
    }

    public boolean isNetworkReady() {
        return !this.isOnline() || this.networkReady;
    }

    public void detachNetwork() {
        try {
            if (this.transport != null) {
                this.transport.close();
            }
        }
        catch (Exception exception) {
        }
        finally {
            this.transport = null;
            this.networkReady = true;
            this.localSide = null;
        }
    }

    public void validateSquareClick(Position pos) {
        if (pos == null) {
            return;
        }
        if (this.boardPanel == null) {
            return;
        }
        if (this.isOnline() && !this.networkReady) {
            ChessFrame.getStatusPanel().setStatus("Waiting for opponent to connect\u2026", Color.GRAY);
            return;
        }
        if (this.gameState.isGameOver()) {
            ChessFrame.getStatusPanel().setStatus("Game is over", Color.GRAY);
            return;
        }
        if (this.isOnline() && this.localSide != null && this.gameState.getTurn() != this.localSide) {
            ChessFrame.getStatusPanel().setStatus("Not your turn", Color.GRAY);
            return;
        }
        this.selectOrMovePiece(pos);
    }

    public void selectOrMovePiece(Position clicked) {
        if (this.selectedPosition == null) {
            this.selectOrIgnore(clicked);
            return;
        }
        if (this.isReselectingOwnPiece(clicked)) {
            this.select(clicked);
            return;
        }
        if (this.isSelectingOwnPieceAfterSelectingEnemy(clicked)) {
            this.select(clicked);
            return;
        }
        this.attemptMove(this.selectedPosition, clicked);
        this.selectedPosition = null;
    }

    @Override
    public void onMessage(String line) {
        this.networkReady = true;
        MoveMessage msg = MoveCodec.tryDecode(line);
        if (msg == null) {
            return;
        }
        SwingUtilities.invokeLater(() -> this.applyNetworkMove(msg));
    }

    @Override
    public void onError(Exception e) {
        e.printStackTrace();
    }

    private void applyNetworkMove(MoveMessage mm) {
        MoveType mt;
        Position to;
        Position from = new Position(mm.fromRank(), mm.fromFile());
        Move legal = MoveGenerator.getValidMoveInTurn(this.gameState, from, to = new Position(mm.toRank(), mm.toFile()));
        if (legal == null) {
            return;
        }
        PromotionPiece promo = null;
        String t = mm.type();
        if (t != null && t.startsWith("PROMOTION")) {
            mt = MoveType.PROMOTION;
            int idx = t.indexOf(58);
            if (idx > 0 && idx + 1 < t.length()) {
                promo = PromotionPiece.valueOf(t.substring(idx + 1));
            }
        } else {
            mt = MoveType.valueOf(t);
        }
        Move toApply = new Move(from, to, mt, promo);
        this.applyMoveAndNotify(toApply, false);
    }

    private void postMoveUiChecks() {
        this.updatePostMoveUi();
        this.maybeShowGameOverDialog();
    }

    private void sendIfOnline(Move m) {
        if (this.transport == null) {
            return;
        }
        String typeStr = m.getMoveType().name();
        if (m.getMoveType() == MoveType.PROMOTION) {
            PromotionPiece pp = m.getPromotion();
            typeStr = pp == null ? "PROMOTION" : "PROMOTION:" + pp.name();
        }
        MoveMessage msg = new MoveMessage(m.getFromPos().getRank(), m.getFromPos().getFile(), m.getToPos().getRank(), m.getToPos().getFile(), typeStr);
        this.transport.send(MoveCodec.encode(msg));
    }

    private void selectOrIgnore(Position p) {
        Piece piece = this.gameState.getPieceAt(p);
        if (piece == null) {
            return;
        }
        this.select(p);
    }

    private boolean isReselectingOwnPiece(Position clicked) {
        Piece from = this.gameState.getPieceAt(this.selectedPosition);
        Piece to = this.gameState.getPieceAt(clicked);
        return from != null && to != null && from.getColor() == to.getColor();
    }

    private boolean isSelectingOwnPieceAfterSelectingEnemy(Position clicked) {
        Piece from = this.gameState.getPieceAt(this.selectedPosition);
        Piece to = this.gameState.getPieceAt(clicked);
        PlayerColor us = this.gameState.getTurn();
        PlayerColor them = us.getOpposite();
        return from != null && to != null && from.getColor() == them && to.getColor() == us;
    }

    private void select(Position p) {
        this.selectedPosition = p;
        if (this.boardPanel != null) {
            if (this.highlightedSquare != null) {
                this.boardPanel.getSquareButton(this.highlightedSquare).setHighlighted(false);
            }
            this.boardPanel.getSquareButton(p).setHighlighted(true);
            this.highlightedSquare = p;
        }
        Piece piece = this.gameState.getPieceAt(p);
        if (this.gameState.isInCheck()) {
            ChessFrame.getStatusPanel().setStatus("Turn: " + String.valueOf((Object)this.gameState.getTurn()), Color.MAGENTA);
        } else {
            ChessFrame.getStatusPanel().setStatus("Turn: " + String.valueOf((Object)this.gameState.getTurn()));
        }
    }

    private void attemptMove(Position from, Position to) {
        Move m = MoveGenerator.getValidMoveInTurn(this.gameState, from, to);
        if (m == null) {
            this.showIllegalMove();
            return;
        }
        if (m.getMoveType() == MoveType.PROMOTION) {
            PromotionPiece choice = PromotionDialog.prompt(this.boardPanel, this.gameState.getTurn());
            if (choice == null) {
                return;
            }
            m = m.withPromotion(choice);
        }
        if (this.engine != null && !this.isOnline()) {
            try {
                this.engine.pushUserMove(this.toUci(m));
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        this.applyMoveAndNotify(m, true);
        this.maybeEngineRespond();
    }

    private void showIllegalMove() {
        ChessFrame.getStatusPanel().setStatus("Illegal move", Color.RED);
        if (this.boardPanel != null && this.highlightedSquare != null) {
            this.boardPanel.getSquareButton(this.highlightedSquare).setHighlighted(false);
            this.highlightedSquare = null;
        }
        this.selectedPosition = null;
    }

    private void applyMoveAndNotify(Move m, boolean broadcast) {
        GameState snap = this.gameState.snapshot();
        String san = NotationSAN.toSAN(this.gameState, m);
        this.history.record(this.gameState, m, snap);
        this.wireLog.add(this.encodeWire(m));
        String line = LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH:mm:ss")) + " " + san;
        this.dispatchMoveInfo(line);
        this.gameState.applyMove(m);
        if (this.boardPanel != null) {
            this.boardPanel.updateBoard(this.gameState.getBoard());
            if (this.highlightedSquare != null) {
                this.boardPanel.getSquareButton(this.highlightedSquare).setHighlighted(false);
                this.highlightedSquare = null;
            }
        }
        this.postMoveUiChecks();
        if (broadcast) {
            this.sendIfOnline(m);
        }
    }

    private void updatePostMoveUi() {
        if (this.gameState.isInCheck()) {
            ChessFrame.getStatusPanel().setStatus(String.valueOf((Object)this.gameState.getTurn()) + " is in check!", new Color(255, 0, 255));
        } else {
            ChessFrame.getStatusPanel().setReady(this.gameState);
        }
    }

    private void maybeShowGameOverDialog() {
        if (!this.gameState.isGameOver()) {
            return;
        }
        String title = "Spiel beendet";
        GameOverType type = this.gameState.getGameOverType();
        PlayerColor winner = this.gameState.getWinner();
        String message = switch (type) {
            case GameOverType.CHECKMATE -> "Checkmate! " + (winner == PlayerColor.WHITE ? "White" : "Black") + " wins.";
            case GameOverType.DRAW -> "Draw.";
            case GameOverType.RESIGN -> (winner == PlayerColor.WHITE ? "White" : "Black") + " resigned. " + (winner == PlayerColor.WHITE ? "White" : "Black") + " wins.";
            default -> "The game is over.";
        };
        JOptionPane.showMessageDialog(null, message, title, 1);
    }

    public void dispatchMoveInfo(String info) {
        ChessFrame.getMoveListPanel().appendMove(info);
        this.moveLog.add(info);
    }

    public void undo() {
        if (!this.history.canUndo()) {
            return;
        }
        this.history.undo(this.gameState);
        if (this.boardPanel != null) {
            this.boardPanel.updateBoard(this.gameState.getBoard());
        }
        this.updatePostMoveUi();
        if (!this.moveLog.isEmpty()) {
            String last = this.moveLog.remove(this.moveLog.size() - 1);
            ChessFrame.getMoveListPanel().removeMove(this.moveLog, last);
        }
    }

    public void redo() {
        if (!this.history.canRedo()) {
            return;
        }
        Move redoMove = this.history.peekRedoMove();
        if (redoMove == null) {
            return;
        }
        String san = NotationSAN.toSAN(this.gameState, redoMove);
        this.history.redo(this.gameState);
        if (this.boardPanel != null) {
            this.boardPanel.updateBoard(this.gameState.getBoard());
        }
        this.updatePostMoveUi();
        String line = LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH:mm:ss")) + " " + san;
        this.dispatchMoveInfo(line);
    }

    private String encodeWire(Move m) {
        Object type = m.getMoveType().name();
        if (m.getMoveType() == MoveType.PROMOTION) {
            type = "PROMOTION" + (String)(m.getPromotion() != null ? ":" + m.getPromotion().name() : "");
        }
        MoveMessage msg = new MoveMessage(m.getFromPos().getRank(), m.getFromPos().getFile(), m.getToPos().getRank(), m.getToPos().getFile(), (String)type);
        return MoveCodec.encode(msg);
    }

    public void startEngineGame(PlayerColor you) {
        this.setLocalSide(you);
        try {
            if (this.engine != null) {
                this.engine.newGame();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (this.engine != null && this.gameState.getTurn() != you) {
            this.engineMoveAsync();
        }
    }

    private void maybeEngineRespond() {
        if (this.engine == null) {
            return;
        }
        if (this.localSide == null) {
            return;
        }
        if (this.isOnline()) {
            return;
        }
        if (this.gameState.getTurn() != this.localSide) {
            this.engineMoveAsync();
        }
    }

    private void engineMoveAsync() {
        if (this.engine == null || this.engineThinking) {
            return;
        }
        this.engineThinking = true;
        new Thread(() -> {
            try {
                String uci = this.engine.bestMoveMs(1500);
                Move em = this.decodeUci(uci);
                if (em != null) {
                    this.engine.pushUserMove(uci);
                    SwingUtilities.invokeLater(() -> this.applyMoveAndNotify(em, false));
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
            finally {
                this.engineThinking = false;
                SwingUtilities.invokeLater(this::maybeEngineRespond);
            }
        }, "engine-move").start();
    }

    private String toUci(Move m) {
        String s = this.sq(m.getFromPos()) + this.sq(m.getToPos());
        if (m.getMoveType() == MoveType.PROMOTION && m.getPromotion() != null) {
            s = s + (switch (m.getPromotion()) {
                default -> throw new MatchException(null, null);
                case PromotionPiece.QUEEN -> "q";
                case PromotionPiece.ROOK -> "r";
                case PromotionPiece.BISHOP -> "b";
                case PromotionPiece.KNIGHT -> "n";
            });
        }
        return s;
    }

    private String sq(Position p) {
        int f = p.getFile();
        int r = p.getRank();
        char file = (char)(97 + f);
        int rank = 8 - r;
        return "" + file + rank;
    }

    private Position parseSquare(String s) {
        int file = s.charAt(0) - 97;
        int rank = s.charAt(1) - 49;
        rank = 7 - rank;
        return new Position(rank, file);
    }

    private Move decodeUci(String uci) {
        Move legal;
        if (uci == null || uci.length() < 4) {
            return null;
        }
        Position from = this.parseSquare(uci.substring(0, 2));
        Position to = this.parseSquare(uci.substring(2, 4));
        PromotionPiece promo = null;
        if (uci.length() >= 5) {
            switch (Character.toLowerCase(uci.charAt(4))) {
                case 'q': {
                    PromotionPiece promotionPiece = PromotionPiece.QUEEN;
                    break;
                }
                case 'r': {
                    PromotionPiece promotionPiece = PromotionPiece.ROOK;
                    break;
                }
                case 'b': {
                    PromotionPiece promotionPiece = PromotionPiece.BISHOP;
                    break;
                }
                case 'n': {
                    PromotionPiece promotionPiece = PromotionPiece.KNIGHT;
                    break;
                }
                default: {
                    PromotionPiece promotionPiece = promo = null;
                }
            }
        }
        if ((legal = MoveGenerator.getValidMoveInTurn(this.gameState, from, to)) == null) {
            return null;
        }
        if (promo != null && legal.getMoveType() == MoveType.PROMOTION) {
            legal = legal.withPromotion(promo);
        }
        return legal;
    }

    static String javaCmd() {
        if (System.getProperty("os.name").toLowerCase().contains("mac")) {
            return "java";
        }
        boolean win = System.getProperty("os.name").toLowerCase().contains("win");
        return Paths.get(System.getProperty("java.home"), "bin", win ? "java.exe" : "java").toString();
    }

    static Path appHome() {
        try {
            Path loc = Paths.get(GameController.class.getProtectionDomain().getCodeSource().getLocation().toURI());
            return Files.isRegularFile(loc, new LinkOption[0]) ? loc.getParent() : loc;
        }
        catch (Exception e) {
            return Paths.get(System.getProperty("user.dir"), new String[0]);
        }
    }
}

