/*
 * Decompiled with CFR 0.152.
 */
package com.frommzay.chess.services.infrastructure.engine;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

public final class UciClient
implements AutoCloseable {
    private final Process proc;
    private final BufferedWriter w;
    private final BufferedReader r;
    private final ExecutorService pump = Executors.newSingleThreadExecutor(r -> {
        Thread t = new Thread(r, "uci-out-pump");
        t.setDaemon(true);
        return t;
    });
    private static final long UCI_HANDSHAKE_TIMEOUT_MS = Long.getLong("uci.handshake.timeout", 10000L);
    private final BlockingQueue<String> lines = new LinkedBlockingQueue<String>(4096);
    private volatile boolean alive = true;
    private final List<String> moveHistory = new ArrayList<String>();
    private String startFEN = "startpos";

    private UciClient(ProcessBuilder pb) throws IOException, TimeoutException {
        pb.redirectErrorStream(true);
        this.proc = pb.start();
        this.w = new BufferedWriter(new OutputStreamWriter(this.proc.getOutputStream(), StandardCharsets.UTF_8));
        this.r = new BufferedReader(new InputStreamReader(this.proc.getInputStream(), StandardCharsets.UTF_8));
        this.pump.submit(() -> {
            try {
                String line;
                while ((line = this.r.readLine()) != null) {
                    this.lines.offer(line);
                }
            }
            catch (IOException iOException) {
            }
            finally {
                this.alive = false;
            }
        });
        this.send("uci");
        this.waitForToken("uciok", UCI_HANDSHAKE_TIMEOUT_MS);
        this.isReady(10000L);
    }

    public static UciClient launchJar(String javaCmd, String jarPath) throws Exception {
        return new UciClient(new ProcessBuilder(javaCmd, "--add-modules=jdk.incubator.vector", "-jar", jarPath));
    }

    public static UciClient launchNative(String enginePath) throws Exception {
        return new UciClient(new ProcessBuilder(enginePath));
    }

    public void isReady(long timeoutMs) throws IOException, TimeoutException {
        this.send("isready");
        this.waitForToken("readyok", timeoutMs);
    }

    public void newGame() throws IOException, TimeoutException {
        this.moveHistory.clear();
        this.startFEN = "startpos";
        this.send("ucinewgame");
        this.isReady(3000L);
    }

    public void setPositionFEN(String fen) throws IOException {
        this.startFEN = "fen " + fen;
        this.moveHistory.clear();
    }

    public void applyUserMove(String uciMove) {
        this.moveHistory.add(uciMove);
    }

    public void setOption(String name, String value) throws IOException, TimeoutException {
        this.send("setoption name " + name + " value " + value);
        this.isReady(2000L);
    }

    public BestMove goMovetime(int ms, long timeoutBufferMs) throws IOException, TimeoutException {
        this.positionSync();
        this.send("go movetime " + ms);
        return this.waitBestMove((long)ms + timeoutBufferMs);
    }

    public BestMove goDepth(int depth, long timeoutMs) throws IOException, TimeoutException {
        this.positionSync();
        this.send("go depth " + depth);
        return this.waitBestMove(timeoutMs);
    }

    public BestMove goNodes(long nodes, long timeoutMs) throws IOException, TimeoutException {
        this.positionSync();
        this.send("go nodes " + nodes);
        return this.waitBestMove(timeoutMs);
    }

    private void positionSync() throws IOException {
        if (this.moveHistory.isEmpty()) {
            this.send("position " + this.startFEN);
        } else {
            this.send("position " + this.startFEN + " moves " + String.join((CharSequence)" ", this.moveHistory));
        }
    }

    private BestMove waitBestMove(long timeoutMs) throws TimeoutException {
        long end = System.currentTimeMillis() + timeoutMs;
        String found = null;
        String ponder = null;
        while (System.currentTimeMillis() < end && this.alive) {
            String line;
            try {
                line = this.lines.poll(20L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                continue;
            }
            if (line == null || !line.startsWith("bestmove ")) continue;
            String[] tok = line.split("\\s+");
            if (tok.length >= 2) {
                found = tok[1];
            }
            if (tok.length < 4 || !"ponder".equals(tok[2])) break;
            ponder = tok[3];
            break;
        }
        if (found == null) {
            throw new TimeoutException("bestmove not received in time");
        }
        return new BestMove(found, ponder);
    }

    private void send(String cmd) throws IOException {
        this.w.write(cmd);
        this.w.write(10);
        this.w.flush();
    }

    private void waitForToken(String token, long timeoutMs) throws TimeoutException {
        long end = System.currentTimeMillis() + timeoutMs;
        while (System.currentTimeMillis() < end && this.alive) {
            String line;
            try {
                line = this.lines.poll(20L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                continue;
            }
            if (line == null || !line.contains(token)) continue;
            return;
        }
        throw new TimeoutException("Timeout waiting for: " + token);
    }

    @Override
    public void close() {
        try {
            this.send("quit");
        }
        catch (Exception exception) {
            // empty catch block
        }
        try {
            this.w.close();
        }
        catch (Exception exception) {
            // empty catch block
        }
        try {
            this.r.close();
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.proc.destroy();
        this.pump.shutdownNow();
    }

    public record BestMove(String move, String ponder) {
    }
}

